/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2012-2014 WaBit Inc. All rights reserved.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.wabit.uecs;

import java.text.DecimalFormat;

/**
 * 共用通信子（CCM）の基底クラスです。
 *
 * @author WaBit
 *
 */
public class Ccm {

    // UECSバージョン属性
    private String uecsVersion = UecsConstants.DEFAULT_VERSION;

    private CcmID type;
    private int room = -1; // -1を未設定と解釈する
    private int region = -1; // -1を未設定と解釈する
    private int order = -1; // -1を未設定と解釈する
    private int priority = -1; // -1を未設定と解釈する
    private char side;
    private CcmLevel level;
    private String unit;
    private Number value;
    private int cast;
    private String format;

    /**
     * 初期化したインスタンスを生成します。
     */
    public Ccm() {
        format = "#0";
        cast = 0;
        unit = "";
    }

    /**
     * 小数点以下有効桁数を取得します。
     *
     * @return 桁数。整数の場合は0
     */
    public int getCast() {
        return cast;
    }

    /**
     * CCM送受信レベルを取得します。
     *
     * @return 送受信レベル値
     */
    public CcmLevel getLevel() {
        return level;
    }

    /**
     * order属性を取得します。
     *
     * @return order値
     */
    public int getOrder() {
        return order;
    }

    /**
     * priority属性を取得します。
     *
     * @return priority値
     */
    public int getPriority() {
        return priority;
    }

    /**
     * region属性を取得します。
     *
     * @return region値
     */
    public int getRegion() {
        return region;
    }

    /**
     * room属性を取得します。
     *
     * @return room値
     */
    public int getRoom() {
        return room;
    }

    /**
     * 送受信方向を取得します。
     *
     * @return 送信側:'S'または受信側:'R'
     */
    public char getSide() {
        return side;
    }

    /**
     * type属性を取得します。
     *
     * @return CCM識別子
     */
    public CcmID getType() {
        return type;
    }

    /**
     * UECS仕様バージョンを取得します。
     *
     * @return バージョン表記文字列
     */
    public String getUecsVersion() {
        return uecsVersion;
    }

    /**
     * 単位を取得します。
     *
     * @return 単位文字列
     */
    public String getUnit() {
        return unit;
    }

    /**
     * CCM値の数値表現を取得します。
     * @return 数値オブジェクト
     */
    public Number getNumberValue() {
        return value;
    }

    /**
     * CCM値の文字列表現を取得します。
     *
     * @return 数値フォーマットされた文字列。内部値がnullの場合は空文字が返ります。
     */
    public String getStringValue() {
        if (value == null) {
            return "";
        }
        return new DecimalFormat(format).format(value);
    }

    /**
     * 数値フォーマットを取得します。
     * @return 数値フォーマット文字列
     */
    public String getFormat() {
        return this.format;
    }

    /**
     * 小数点以下有効桁数を設定します。
     *
     * @param cast 0以上の整数
     */
    public void setCast(int cast) {
        if (cast <= 0) {
            format = "#0";
        } else {
            StringBuilder sb = new StringBuilder("#0.");
            for (int i = 0; i < cast; i++) {
                sb.append('0');
            }
            format = sb.toString();
        }
        this.cast = cast;
    }

    /**
     * CCM送受信レベルを設定します。
     * Bレベルが設定されると、自動的にpriorityは30に設定されます。
     *
     * @param level 送受信レベル
     */
    public void setLevel(CcmLevel level) {
        this.level = level;
        if (level != null && level.getCategory() == UecsConstants.LEVEL_B) {
            setPriority(30);
        }
    }

    /**
     * order属性を設定します。
     *
     * @param order 0から30000までの数値
     */
    public void setOrder(int order) {
        if (order < 0 || order > 30000) {
            throw new NodeRuntimeException("order(0 to 30000) : " + order);
        }
        this.order = order;
    }

    /**
     * priority属性を設定します。
     *
     * @param priority 0から30までの数値
     */
    public void setPriority(int priority) {
        if (priority < 0 || priority > 30) {
            throw new NodeRuntimeException("priority(0 to 30) : " + priority);
        }
        this.priority = priority;
    }

    /**
     * region属性を設定します。
     *
     * @param region  0から127までの数値
     */
    public void setRegion(int region) {
        if (region < 0 || region > 127) {
            throw new NodeRuntimeException("region(0 to 127) : " + region);
        }
        this.region = region;
    }

    /**
     * room属性を設定します。
     *
     * @param room  0から127までの数値
     */
    public void setRoom(int room) {
        if (room < 0 || room > 127) {
            throw new NodeRuntimeException("room(0 to 127) : " + room);
        }
        this.room = room;
    }

    /**
     * 送受信方向を設定します。
     *
     * @param side 送信側:'S'または受信側:'R'
     */
    public void setSide(char side) {
        if (side != UecsConstants.SENDER && side != UecsConstants.RECEIVER) {
            throw new NodeRuntimeException("side (S or R) : " + side);
        }
        this.side = side;
    }

    /**
     * type属性を設定します。
     *
     * @param type CCM識別子
     */
    public void setType(CcmID type) {
        this.type = type;
    }

    /**
     * UECS仕様バージョンを設定します。
     *
     * @param uecsVersion バージョン表記文字列
     */
    public void setUecsVersion(String uecsVersion) {
        this.uecsVersion = uecsVersion;
    }

    /**
     * 単位を設定します。
     *
     * @param unit 単位文字列
     */
    public void setUnit(String unit) {
        this.unit = unit;
    }

    /**
     * 値を設定します。
     *
     * @param number 数値
     */
    protected void setValue(Number number) {
        this.value = number;
    }

    /**
     * 値を文字列表現で設定します。
     *
     * @param value 空文字の場合はnullがセットされます。
     */
    protected void setValue(String value) {
        if (value == null || value.length() == 0) {
            this.value = null;
            return;
        }
        try {
            this.value = new DecimalFormat(format).parse(value);
        } catch (Throwable t) {
            throw new NodeRuntimeException("value format error. format="
                    + format + ", value=" + value, t);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || !(o instanceof Ccm)) {
            return false;
        }
        //TODO 属性値で比較が必要か？
        return false;
    }
}
